<?php

require_once '../core/Controller.php';

class AgentController extends Controller
{
    private $agentModel;
    private $transactionModel;

    public function __construct()
    {
        parent::__construct();
        $this->agentModel = $this->loadModel('Agent');
        $this->transactionModel = $this->loadModel('Transaction');
    }

    public function index()
    {
        $this->requireAuth();
        
        $currentAgent = $_SESSION['admin'];
        
        if ($currentAgent['agent_level'] === 'admin') {
            $agents = $this->agentModel->getAllAgentsForAdmin();
        } else {
            $agents = $this->agentModel->getSubAgents($currentAgent['agent_code']);
        }
        
        return $this->view->render('agents.index', [
            'title' => 'Agent Management',
            'agents' => $agents,
            'currentAgent' => $currentAgent,
            'flash' => $this->getFlashMessage()
        ]);
    }

    public function create()
    {
        $this->requireAuth();
        
        return $this->view->render('agents.create', [
            'title' => 'Add New Agent',
            'flash' => $this->getFlashMessage()
        ]);
    }

    public function store()
    {
        $this->requireAuth();
        
        if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
            $this->redirect('/agents');
        }

        $currentAgent = $_SESSION['admin'];
        $agentName = $_POST['agent_name'] ?? '';
        $password = $_POST['password'] ?? '';

        if (empty($agentName) || empty($password)) {
            $this->setFlashMessage('error', 'Error', 'All fields are required');
            $this->redirect('/agents/create');
        }

        if ($this->agentModel->findBy('agent_name', $agentName)) {
            $this->setFlashMessage('error', 'Error', 'Agent name already exists');
            $this->redirect('/agents/create');
        }

        $agentData = [
            'agent_code' => $this->generateAgentCode(),
            'agent_name' => $agentName,
            'agent_token' => md5(uniqid()),
            'agent_password' => md5($password),
            'agent_upline' => $currentAgent['agent_code'],
            'agent_created' => date('Y-m-d H:i:s'),
            'agent_balance' => 0,
            'agent_level' => 'agent',
            'agent_status' => 'active'
        ];

        if ($this->agentModel->create($agentData)) {
            $this->setFlashMessage('success', 'Success', 'Agent created successfully');
        } else {
            $this->setFlashMessage('error', 'Error', 'Failed to create agent');
        }

        $this->redirect('/agents');
    }

    public function edit($params)
    {
        $this->requireAuth();
        
        $agent = $this->agentModel->find($params['id']);
        if (!$agent) {
            $this->setFlashMessage('error', 'Error', 'Agent not found');
            $this->redirect('/agents');
        }

        return $this->view->render('agents.edit', [
            'title' => 'Edit Agent',
            'agent' => $agent,
            'flash' => $this->getFlashMessage()
        ]);
    }

    public function update($params)
    {
        $this->requireAuth();
        
        if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
            $this->redirect('/agents');
        }

        $password = $_POST['password'] ?? '';
        
        if (empty($password)) {
            $this->setFlashMessage('error', 'Error', 'Password is required');
            $this->redirect("/agents/{$params['id']}/edit");
        }

        if ($this->agentModel->update($params['id'], ['agent_password' => md5($password)])) {
            $this->setFlashMessage('success', 'Success', 'Agent password updated successfully');
        } else {
            $this->setFlashMessage('error', 'Error', 'Failed to update agent');
        }

        $this->redirect('/agents');
    }

    public function updateBalance($params)
    {
        $this->requireAuth();
        
        if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
            return $this->json(['status' => 'error', 'message' => 'Invalid request method']);
        }

        $currentAgent = $_SESSION['admin'];
        $targetAgent = $this->agentModel->find($params['id']);
        $action = $_POST['action'] ?? '';
        $amount = (float)($_POST['amount'] ?? 0);

        if (!$targetAgent || $amount <= 0) {
            return $this->json(['status' => 'error', 'message' => 'Invalid parameters']);
        }

        $beforeBalance = $currentAgent['agent_balance'];

        if ($action === 'add') {
            if ($beforeBalance < $amount) {
                return $this->json(['status' => 'error', 'message' => 'Insufficient balance']);
            }
            
            $this->agentModel->subtractBalance($currentAgent['agent_code'], $amount);
            $this->agentModel->addBalance($targetAgent['agent_code'], $amount);
            
            $afterBalance = $beforeBalance - $amount;
        } else {
            if ($targetAgent['agent_balance'] < $amount) {
                return $this->json(['status' => 'error', 'message' => 'Insufficient target agent balance']);
            }
            
            $this->agentModel->subtractBalance($targetAgent['agent_code'], $amount);
            $this->agentModel->addBalance($currentAgent['agent_code'], $amount);
            
            $afterBalance = $beforeBalance + $amount;
        }

        // Record transaction
        $this->transactionModel->create([
            'trxid' => md5(uniqid()),
            'to_agent' => $action === 'add' ? $targetAgent['agent_code'] : $currentAgent['agent_code'],
            'from_agent' => $action === 'add' ? $currentAgent['agent_code'] : $targetAgent['agent_code'],
            'action' => $action,
            'amount' => $amount,
            'before_agent_balance' => $beforeBalance,
            'after_agent_balance' => $afterBalance,
            'note' => $action === 'add' ? 'Transfer Out' : 'Transfer In',
            'created' => date('Y-m-d H:i:s')
        ]);

        return $this->json([
            'status' => 'success',
            'message' => 'Balance updated successfully',
            'data' => [
                'before_balance' => $beforeBalance,
                'after_balance' => $afterBalance
            ]
        ]);
    }

    public function blockAgent($params)
    {
        $this->requireAuth();
        
        if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
            return $this->json(['status' => 'error', 'message' => 'Invalid request method']);
        }

        $agent = $this->agentModel->find($params['id']);
        if (!$agent) {
            return $this->json(['status' => 'error', 'message' => 'Agent not found']);
        }

        if ($this->agentModel->update($params['id'], ['agent_status' => 'blocked'])) {
            return $this->json([
                'status' => 'success',
                'message' => 'Agent blocked successfully'
            ]);
        }

        return $this->json(['status' => 'error', 'message' => 'Failed to block agent']);
    }

    public function unblockAgent($params)
    {
        $this->requireAuth();
        
        if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
            return $this->json(['status' => 'error', 'message' => 'Invalid request method']);
        }

        $agent = $this->agentModel->find($params['id']);
        if (!$agent) {
            return $this->json(['status' => 'error', 'message' => 'Agent not found']);
        }

        if ($this->agentModel->update($params['id'], ['agent_status' => 'active'])) {
            return $this->json([
                'status' => 'success',
                'message' => 'Agent unblocked successfully'
            ]);
        }

        return $this->json(['status' => 'error', 'message' => 'Failed to unblock agent']);
    }

    private function generateAgentCode()
    {
        $characters = '0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZ';
        do {
            $code = '';
            for ($i = 0; $i < 8; $i++) {
                $code .= $characters[rand(0, strlen($characters) - 1)];
            }
        } while ($this->agentModel->findBy('agent_code', $code));
        
        return $code;
    }
}