<?php
// app/Services/NexusGamingApi.php - Nexus Gaming API Integration

class NexusGamingApi
{
    private $apiUrl = 'https://api.nexusggr.com';
    private $agentCode;
    private $agentToken;
    
    public function __construct($agentCode = '', $agentToken = '')
    {
        $this->agentCode = $agentCode;
        $this->agentToken = $agentToken;
    }
    
    // 1. Create User
    public function createUser($userCode)
    {
        $data = [
            'method' => 'user_create',
            'agent_code' => $this->agentCode,
            'agent_token' => $this->agentToken,
            'user_code' => $userCode
        ];
        
        return $this->makeRequest($data);
    }
    
    // 2. User Deposit
    public function userDeposit($userCode, $amount, $agentSign = null)
    {
        $data = [
            'method' => 'user_deposit',
            'agent_code' => $this->agentCode,
            'agent_token' => $this->agentToken,
            'user_code' => $userCode,
            'amount' => $amount
        ];
        
        if ($agentSign) {
            $data['agent_sign'] = $agentSign;
        }
        
        return $this->makeRequest($data);
    }
    
    // 3. User Withdraw
    public function userWithdraw($userCode, $amount, $agentSign = null)
    {
        $data = [
            'method' => 'user_withdraw',
            'agent_code' => $this->agentCode,
            'agent_token' => $this->agentToken,
            'user_code' => $userCode,
            'amount' => $amount
        ];
        
        if ($agentSign) {
            $data['agent_sign'] = $agentSign;
        }
        
        return $this->makeRequest($data);
    }
    
    // 4. Reset User Balance
    public function resetUserBalance($userCode = null, $allUsers = false)
    {
        $data = [
            'method' => 'user_withdraw_reset',
            'agent_code' => $this->agentCode,
            'agent_token' => $this->agentToken
        ];
        
        if ($allUsers) {
            $data['all_users'] = true;
        } elseif ($userCode) {
            $data['user_code'] = $userCode;
        }
        
        return $this->makeRequest($data);
    }
    
    // 5. Launch Game
    public function launchGame($userCode, $providerCode, $gameCode, $lang = 'en')
    {
        $data = [
            'method' => 'game_launch',
            'agent_code' => $this->agentCode,
            'agent_token' => $this->agentToken,
            'user_code' => $userCode,
            'provider_code' => $providerCode,
            'game_code' => $gameCode,
            'lang' => $lang
        ];
        
        return $this->makeRequest($data);
    }
    
    // 6. Get Money Info (Agent/User Balance)
    public function getMoneyInfo($userCode = null, $allUsers = false)
    {
        $data = [
            'method' => 'money_info',
            'agent_code' => $this->agentCode,
            'agent_token' => $this->agentToken
        ];
        
        if ($allUsers) {
            $data['all_users'] = true;
        } elseif ($userCode) {
            $data['user_code'] = $userCode;
        }
        
        return $this->makeRequest($data);
    }
    
    // 7. Get Provider List
    public function getProviderList()
    {
        $data = [
            'method' => 'provider_list',
            'agent_code' => $this->agentCode,
            'agent_token' => $this->agentToken
        ];
        
        return $this->makeRequest($data);
    }
    
    // 8. Get Game List
    public function getGameList($providerCode)
    {
        $data = [
            'method' => 'game_list',
            'agent_code' => $this->agentCode,
            'agent_token' => $this->agentToken,
            'provider_code' => $providerCode
        ];
        
        return $this->makeRequest($data);
    }
    
    // 9. Get Game History
    public function getGameLog($userCode, $gameType, $start, $end, $page = 0, $perPage = 1000)
    {
        $data = [
            'method' => 'get_game_log',
            'agent_code' => $this->agentCode,
            'agent_token' => $this->agentToken,
            'user_code' => $userCode,
            'game_type' => $gameType,
            'start' => $start,
            'end' => $end,
            'page' => $page,
            'perPage' => $perPage
        ];
        
        return $this->makeRequest($data);
    }
    
    // 10. Transfer Status
    public function getTransferStatus($userCode, $agentSign)
    {
        $data = [
            'method' => 'transfer_status',
            'agent_code' => $this->agentCode,
            'agent_token' => $this->agentToken,
            'user_code' => $userCode,
            'agent_sign' => $agentSign
        ];
        
        return $this->makeRequest($data);
    }
    
    // Call API - Get Current Playing Users
    public function getCallPlayers()
    {
        $data = [
            'method' => 'call_players',
            'agent_code' => $this->agentCode,
            'agent_token' => $this->agentToken
        ];
        
        return $this->makeRequest($data);
    }
    
    // Call API - Get Call List
    public function getCallList($providerCode, $gameCode, $userCode)
    {
        $data = [
            'method' => 'call_list',
            'agent_code' => $this->agentCode,
            'agent_token' => $this->agentToken,
            'provider_code' => $providerCode,
            'game_code' => $gameCode,
            'user_code' => $userCode
        ];
        
        return $this->makeRequest($data);
    }
    
    // Call API - Apply Call
    public function applyCall($providerCode, $gameCode, $userCode, $callRtp, $callType = 1)
    {
        $data = [
            'method' => 'call_apply',
            'agent_code' => $this->agentCode,
            'agent_token' => $this->agentToken,
            'provider_code' => $providerCode,
            'game_code' => $gameCode,
            'user_code' => $userCode,
            'call_rtp' => $callRtp,
            'call_type' => $callType
        ];
        
        return $this->makeRequest($data);
    }
    
    // Call API - Get Call History
    public function getCallHistory($offset = 0, $limit = 100)
    {
        $data = [
            'method' => 'call_history',
            'agent_code' => $this->agentCode,
            'agent_token' => $this->agentToken,
            'offset' => $offset,
            'limit' => $limit
        ];
        
        return $this->makeRequest($data);
    }
    
    // Call API - Cancel Call
    public function cancelCall($callId)
    {
        $data = [
            'method' => 'call_cancel',
            'agent_code' => $this->agentCode,
            'agent_token' => $this->agentToken,
            'call_id' => $callId
        ];
        
        return $this->makeRequest($data);
    }
    
    // Control User RTP
    public function controlUserRtp($providerCode, $userCode, $rtp)
    {
        $data = [
            'method' => 'control_rtp',
            'agent_code' => $this->agentCode,
            'agent_token' => $this->agentToken,
            'provider_code' => $providerCode,
            'user_code' => $userCode,
            'rtp' => $rtp
        ];
        
        return $this->makeRequest($data);
    }
    
    // Control Multiple Users RTP
    public function controlUsersRtp($userCodes, $rtp)
    {
        $data = [
            'method' => 'control_users_rtp',
            'agent_code' => $this->agentCode,
            'agent_token' => $this->agentToken,
            'user_codes' => json_encode($userCodes),
            'rtp' => $rtp
        ];
        
        return $this->makeRequest($data);
    }
    
    private function makeRequest($data)
    {
        $ch = curl_init();
        $jsonData = json_encode($data);
        
        curl_setopt($ch, CURLOPT_URL, $this->apiUrl);
        curl_setopt($ch, CURLOPT_POST, 1);
        curl_setopt($ch, CURLOPT_POSTFIELDS, $jsonData);
        curl_setopt($ch, CURLOPT_HTTPHEADER, ['Content-Type: application/json']);
        curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, 5);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
        
        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);
        
        if ($response === false || $httpCode !== 200) {
            return [
                'status' => 0,
                'msg' => 'API_CONNECTION_ERROR'
            ];
        }
        
        return json_decode($response, true);
    }
}