<?php
// cron/sync-games.php - Auto sync games from external API

require_once '../core/Database.php';
require_once '../app/Models/Game.php';
require_once '../app/Models/Provider.php';

// Set timezone
date_default_timezone_set('Asia/Jakarta');

// Initialize models
$gameModel = new Game();
$providerModel = new Provider();

// Log start
echo "[" . date('Y-m-d H:i:s') . "] Starting game sync...\n";

try {
    // Get real game data from Nexus Gaming API
    require_once '../app/Services/NexusGamingApi.php';
    
    // Get admin agent for API calls
    $stmt = $db->prepare("SELECT * FROM tb_agent WHERE agent_level = 'admin' LIMIT 1");
    $stmt->execute();
    $adminAgent = $stmt->fetch();
    
    if (!$adminAgent) {
        throw new Exception("Admin agent not found");
    }
    
    $api = new NexusGamingApi($adminAgent['agent_code'], $adminAgent['agent_token']);
    
    // Get providers first
    $providersResponse = $api->getProviderList();
    $externalGames = [];
    
    if ($providersResponse['status'] == 1) {
        foreach ($providersResponse['providers'] as $provider) {
            // Get games for each provider
            $gamesResponse = $api->getGameList($provider['code']);
            
            if ($gamesResponse['status'] == 1) {
                foreach ($gamesResponse['games'] as $game) {
                    $externalGames[] = [
                        'game_code' => $game['game_code'],
                        'game_name' => $game['game_name'],
                        'game_provider' => $provider['code'],
                        'game_type' => 'slot',
                        'game_image' => $game['banner'],
                        'game_status' => $game['status']
                    ];
                }
            }
        }
    }

    $syncedCount = 0;
    foreach ($externalGames as $gameData) {
        if ($gameModel->syncGame($gameData)) {
            $syncedCount++;
            echo "[" . date('Y-m-d H:i:s') . "] Synced: " . $gameData['game_name'] . "\n";
        }
    }

    echo "[" . date('Y-m-d H:i:s') . "] Game sync completed. Synced: {$syncedCount} games\n";

} catch (Exception $e) {
    echo "[" . date('Y-m-d H:i:s') . "] ERROR: " . $e->getMessage() . "\n";
    
    // Log to file
    file_put_contents('../logs/sync-games.log', date('Y-m-d H:i:s') . " ERROR: " . $e->getMessage() . "\n", FILE_APPEND);
}